#!/usr/bin/env python3
"""Generate a simple report for the mass‑gap simulation.

This script reads the ensemble‑averaged mass gap summary produced by
``run.py``, computes the average effective mass as a function of the
lattice size and performs a linear fit in ``1/L`` to estimate the
continuum limit.  A scatter plot and fitted line are saved as a PNG
and a Markdown report summarising the results is written to
``REPORT_mass_gap_final.md`` in the ``vol4_mass_gap_sim`` directory.
"""

from __future__ import annotations

import os
from pathlib import Path
import matplotlib
matplotlib.use('Agg')  # Use a non‑interactive backend
import matplotlib.pyplot as plt
import numpy as np
import pandas as pd
import yaml


def main() -> None:
    # Resolve repository paths
    script_path = Path(__file__).resolve()
    module_dir = script_path.parent.parent  # vol4_mass_gap_sim
    repo_root = module_dir.parent           # fphs
    # Input summary CSV
    summary_csv = repo_root / 'data' / 'results' / 'vol4_mass_gap_sim' / 'mass_gap_summary.csv'
    if not summary_csv.exists():
        raise FileNotFoundError(f"Summary CSV not found: {summary_csv}")
    df = pd.read_csv(summary_csv)
    # Compute mean effective mass by lattice size
    grouped = df.groupby('L')['m_eff'].mean().sort_index()
    L_vals = grouped.index.to_numpy(dtype=float)
    m_vals = grouped.values.astype(float)
    inv_L = 1.0 / L_vals
    # Linear fit m(L) = slope*(1/L) + intercept
    coeffs = np.polyfit(inv_L, m_vals, 1)
    slope, intercept = coeffs
    fit_m = slope * inv_L + intercept
    # Create plot
    fig, ax = plt.subplots(figsize=(5, 3.5))
    ax.scatter(inv_L, m_vals, label='Simulation', color='C0')
    ax.plot(inv_L, fit_m, label=f'Fit: m(L)=%.3e*(1/L)+%.3e' % (slope, intercept), color='C1')
    ax.set_xlabel('1/L')
    ax.set_ylabel('Mean m_eff')
    ax.set_title('Effective mass vs inverse lattice size')
    ax.legend()
    fig.tight_layout()
    plot_path = module_dir / 'mass_gap_fit.png'
    fig.savefig(plot_path)
    plt.close(fig)
    # Write Markdown report
    report_path = module_dir / 'REPORT_mass_gap_final.md'
    with open(report_path, 'w', encoding='utf-8') as fh:
        fh.write('# Mass Gap Simulation Report\n\n')
        fh.write('This report summarises the results of the mass–gap simulation run. '\
                 'The table below shows the mean effective mass as a function '\
                 'of lattice size, along with a linear extrapolation to the continuum limit.\n\n')
        # Write table of mean m_eff by L
        fh.write('## Ensemble‐averaged effective mass\n\n')
        fh.write('| L | 1/L | mean m_eff |\n')
        fh.write('|---|----|-----------:|\n')
        for L, m in zip(L_vals, m_vals):
            fh.write(f'| {int(L)} | {1.0/L:.3f} | {m:.6f} |\n')
        fh.write('\n')
        fh.write(f'Linear fit: $m(L) = {slope:.3e}\,L^{{-1}} + {intercept:.3e}$\n\n')
        fh.write(f'Predicted continuum limit $m(\infty) = {intercept:.6f}$.\n\n')
        fh.write('## Plot\n\n')
        # Embed image relative to report
        img_rel = os.path.relpath(plot_path, start=module_dir)
        fh.write(f'![Mass gap fit]({img_rel})\n')
    print(f'Report written to {report_path}')


if __name__ == '__main__':
    main()